<?php

require '../../config/config.php'; // Conexión a MySQLi

if ($_SERVER["REQUEST_METHOD"] !== "POST") {
    respuestaJSON("error", "Método no permitido.");
}

$idArchivo  = $_POST['idArchivo'] ?? null;
$usuarioID  = $_POST['usuarioID'] ?? null;
$fechaPago  = $_POST['fechaPago'] ?? null;
$nuevoArchivo = $_FILES['nuevoArchivo'] ?? null;

if (!$idArchivo || !$usuarioID) {
    respuestaJSON("error", "Faltan datos obligatorios.");
}

$query = "SELECT id_empleado, nombre_archivo, ruta_archivo, fechaPago FROM tbl_archivos WHERE id_archivo = ?";
$stmt = $conexion->prepare($query);
$stmt->bind_param("i", $idArchivo);
$stmt->execute();
$result = $stmt->get_result();
$archivoActual = $result->fetch_assoc();

if (!$archivoActual) {
    respuestaJSON("error", "El archivo no existe.");
}

if ($archivoActual['id_empleado'] != $usuarioID) {
    respuestaJSON("error", "No tienes permisos para modificar este archivo.");
}

$updateFields = [];
$params = [];
$types = "";

$fechaPagoNueva = !empty($fechaPago) ? date('Y-m-d', strtotime($fechaPago)) : $archivoActual['fechaPago'];
$fechaPagoAntigua = date('Y-m-d', strtotime($archivoActual['fechaPago']));
$moverArchivo = ($fechaPagoNueva !== $fechaPagoAntigua);

$query = "SELECT user_rfc  FROM tbl_empleados WHERE id_empleado = ?";
$stmt = $conexion->prepare($query);
$stmt->bind_param("i", $usuarioID);
$stmt->execute();
$result = $stmt->get_result();
$empleado = $result->fetch_assoc();

if (!$empleado) {
    respuestaJSON("error", "El empleado no existe.");
}

$rfcuser = $empleado['user_rfc'];

// Extraer Año, Mes, Día de la nueva fecha de pago
list($year, $month, $day) = explode('-', $fechaPagoNueva);
$uploadDirNuevo = "uploads/{$year}/{$month}/{$day}/";

// Crear carpeta si no existe
if (!is_dir($uploadDirNuevo)) {
    mkdir($uploadDirNuevo, 0777, true);
}

try {
    $conexion->begin_transaction();

    if ($nuevoArchivo && $nuevoArchivo['error'] === UPLOAD_ERR_OK) {
        $maxFileSize = 5 * 1024 * 1024; // 5MB
        $fileTmpPath = $nuevoArchivo['tmp_name'];
        $fileName = $nuevoArchivo['name'];
        $fileSize = $nuevoArchivo['size'];

        if ($fileSize > $maxFileSize) {
            respuestaJSON("error", "El archivo excede 5MB.");
        }

        $extension = pathinfo($fileName, PATHINFO_EXTENSION);

        // Verificar que el archivo sea ZIP
        if ($extension !== 'zip') {
            respuestaJSON("error", "Solo se permiten archivos ZIP.");
        }

        $timestamp = time();
        $newFileName = "{$rfcuser}_" . str_replace("-", "", $fechaPagoNueva) . "_{$timestamp}.{$extension}";

        $newFilePath = $uploadDirNuevo . $newFileName;

        // Eliminar el archivo anterior si existe
        if (file_exists($archivoActual['ruta_archivo'])) {
            unlink($archivoActual['ruta_archivo']);
        }

        // Mover el archivo nuevo
        if (!move_uploaded_file($fileTmpPath, $newFilePath)) {
            throw new Exception("Error al mover el archivo.");
        }

        $updateFields[] = "nombre_archivo = ?, ruta_archivo = ?";
        $params[] = $newFileName;
        $params[] = $newFilePath;
        $types .= "ss";
    } elseif ($moverArchivo) {
        // Si solo es mover el archivo por la nueva fecha
        $oldFilePath = $archivoActual['ruta_archivo'];
        $oldFileName = basename($oldFilePath);

        // Extraer la extensión del archivo original
        $extension = pathinfo($oldFileName, PATHINFO_EXTENSION);

        // Generar nuevo nombre basado en la nueva fecha de pago
        $timestamp = time();
        $newFileName = "{$rfcuser}_" . str_replace("-", "", $fechaPagoNueva) . "_{$timestamp}.{$extension}";
        $newFilePath = $uploadDirNuevo . $newFileName;

        // Mover y renombrar el archivo
        if (!rename($oldFilePath, $newFilePath)) {
            throw new Exception("Error al mover y renombrar el archivo.");
        }

        $updateFields[] = "nombre_archivo = ?, ruta_archivo = ?";
        $params[] = $newFileName;
        $params[] = $newFilePath;
        $types .= "ss";
    }

    if (!empty($fechaPago)) {
        $updateFields[] = "fechaPago = ?";
        $params[] = $fechaPagoNueva;
        $types .= "s";
    }

    if (empty($updateFields)) {
        respuestaJSON("error", "No se enviaron cambios.");
    }

    $updateSQL = "UPDATE tbl_archivos SET " . implode(", ", $updateFields) . " WHERE id_archivo = ?";
    $params[] = $idArchivo;
    $types .= "i";

    $stmt = $conexion->prepare($updateSQL);
    $stmt->bind_param($types, ...$params);

    if ($stmt->execute()) {
        $conexion->commit();

        // Eliminar carpetas vacías si no hay más archivos
        $oldDir = dirname($archivoActual['ruta_archivo']);
        while (is_dir($oldDir) && is_writable($oldDir)) {
            $files = array_diff(scandir($oldDir), ['.', '..']);
            if (empty($files)) {
                rmdir($oldDir);
                $oldDir = dirname($oldDir); // Subir un nivel
            } else {
                break;
            }
        }

        respuestaJSON("success", "Archivo actualizado.", ["ruta" => $newFilePath ?? $archivoActual['ruta_archivo']]);
    } else {
        $conexion->rollback();
        respuestaJSON("error", "Error en la actualización.");
    }
} catch (Exception $e) {
    $conexion->rollback();
    respuestaJSON("error", $e->getMessage());
}

function respuestaJSON($status, $message, $data = [])
{
    echo json_encode(array_merge(["status" => $status, "message" => $message], $data));
    exit;
}